#
# TAS API MQTT 連線範例程式
# 程式中使用 certifi 套件提供的 SSL 根憑證，以避免 TAS API SSL 連線憑證更換時造成連線失敗
#
# 程式使用方式：
#    用於配合 TAS API 從租用服務號碼 callout 或是 call-in 到租用服務號碼時，播放語音與回應用戶端輸入的數字按鈕
#    配合 call-in 模式時，要預先使用 phoneConfig API 輸入以下設定資料：
#
#{
#    "serviceNumber": "02819XXXXX",
#    "ivrData": [
#        {
#            "node": "MAIN",
#            "nextNode": "CUSTOM"
#        }
#    ]
#}
#
# 執行環境需求：
#    安裝套件 paho.mqtt 2.0 以上版本，從 1.X 升級到 2.X 的更動說明 https://eclipse.dev/paho/files/paho.mqtt.python/html/migrations.html
#    pip install 'paho-mqtt>=2.0' --upgrade
#    pip install certifi --upgrade
#

import paho.mqtt.client as mqtt
import certifi
import json

brokerUrl = "tasapi.cht.com.tw"
brokerPort = 2883
APIKey = "YOUR_API_KEY"
SNKey = "YOUR_SNKEY"

calloutResultTopic = "phone-conn/calloutResult/" + SNKey
callEventTopic = "phone-conn/callEvent/" + SNKey
callActionTopic = "phone-conn/callAction/" + SNKey
callActionDebugTopic = "phone-conn/callActionDebug/" + SNKey

def print_config():
    print(f"brokerUrl: {brokerUrl}")
    print(f"brokerPort: {brokerPort}")
    print(f"APIKey: {APIKey}")
    print(f"SNKey: {SNKey}")
    
    print(f"calloutResultTopic: {calloutResultTopic}")
    print(f"callEventTopic: {callEventTopic}")
    print(f"callActionTopic: {callActionTopic}")
    print(f"callActionDebugTopic: {callActionDebugTopic}")

    print(f"certifi.where(): {certifi.where()}")

# 連線成功回呼函數
def on_connect(client, userdata, flags, reason_code, properties):
    print(f"Connected with result code {reason_code}")
    # 在連線成功後訂閱一個主題
    client.subscribe(callEventTopic)
    client.subscribe(callActionDebugTopic)

# 收到訊息回呼函數
def on_message(client, userdata, msg):
    # 處理所有從 MQTT topic 收到的訊息
    print(f"{msg.topic}, received msg: {msg.payload.decode()}")

    # 依 topic 處理訊息
    if msg.topic == callEventTopic:
        print(f"callEventTopic, received msg: {msg.payload.decode()}")
        handle_call_event_msg(msg)
    elif msg.topic == calloutResultTopic:
        print(f"calloutResultTopic, received msg: {msg.payload.decode()}")
    else:
        print(f"callActionDebugTopic, received msg: {msg.payload.decode()}")
    
def publish_msg(client, json_resp):
    # 發布訊息到指定 topic
    message = json.dumps(json_resp, ensure_ascii=False)
    client.publish(callActionTopic, message)
    print(f"Published message: {message}")

def handle_call_event_msg(msg):
    # 處理從 callEvent topic 收到的訊息
    req = json.loads(msg.payload.decode())
    msg_type = req.get("type")
    print(f"msg_type: {msg_type}")

    if req.get("type") in ["request", "calloutRequest"]:
        node = req.get("node")
        if node == "MAIN":
            handle_main(req)
        elif node.startswith("askQuestion"):
            user_input = node.split(",")[1]
            print(f"user_input: {user_input}")
            # 依用戶按鍵輸入送回不同回應訊息
            if user_input == "1":
                input_dtmf1(req)
            elif user_input == "2":
                input_dtmf2(req)
            elif user_input == "timeout":
                input_timeout(req)
            else:
                input_other(req)
        else:
            print(f"error, node: {node}")
    

def handle_main(req):
    # 回應的 JSON
    # id 請設定為最近一次從 callEvent topic 中收到訊訊的 id 值
    resp = {
        "id": req.get("id"),
        "node": "askQuestion",
        "collectDTMF": True,
        "text": "今天天氣如何，晴天請按 1，雨天請按 2",
        "promptMode": "F",
        "collectDTMFTimeout": "10",
        "nextNode": "CUSTOM"
    }

    # 發送回應訊息
    publish_msg(client, resp)
    
def input_dtmf1(req):
    # 回應的 JSON
    resp = {
        "id": req.get("id"),
        "node": "playDTMF1",
        "text": "好天氣，出去走走吧",
        "promptMode": "F",
        "nextNode": "END"
    }

    # 發送回應訊息
    publish_msg(client, resp)
 
def input_dtmf2(req):
    resp = {
        "id": req.get("id"),
        "node": "playDTMF2",
        "text": "下雨天，在家看書休息",
        "collectDTMF": True,
        "promptMode": "F",
        "nextNode": "END"
    }

    publish_msg(client, resp)

def input_other(req):
    resp = {
        "id": req.get("id"),
        "node": "askQuestion",
        "text": "輸入數字錯誤，請重新輸入。今天天氣如何，晴天請按 1，雨天請按 2",
        "collectDTMF": True,
        "promptMode": "F",
        "collectDTMFTimeout": "10",
        "nextNode": "CUSTOM"
    }

    publish_msg(client, resp)


def input_timeout(req):
    resp = {
        "id": req.get("id"),
        "node": "askQuestion",
        "text": "沒有收到您的輸入，今天天氣如何，晴天請按 1，雨天請按 2",
        "collectDTMF": True,
        "promptMode": "F",
        "collectDTMFTimeout": "10",
        "nextNode": "CUSTOM"
    }

    publish_msg(client, resp)

# print program config
print_config()

# 建立 MQTT 客戶端
client = mqtt.Client(mqtt.CallbackAPIVersion.VERSION2, reconnect_on_failure = True)

# 設定回呼函數
client.on_connect = on_connect
client.on_message = on_message

# 設定 TLS/SSL
client.tls_set(certifi.where())

client.username = APIKey
client.password = APIKey

# 使用 TLS/SSL 連線到 MQTT 伺服器, keepalive 設定為 60 秒
client.connect("tasapi.cht.com.tw", 2883, 60)

# 開始循環以處理網路流量和回呼
client.loop_forever()

